//**********************************************************************************************
//* Mail Reader Example.
//* --------------------
//*	Demonstrates: 
//*					- WebMail EMail/POP3/SMTP/IMAP encapsulation classes.
//* 				- simlpe multithreading.
//*					- serialization.
//*				
//* Currently, only POP3 class is implemented.
//*


#include "MailReader.h"

#define IMAGEFILE <MailReader/MailReader.iml>
#define IMAGECLASS MailImages 
#include <Draw/iml_source.h>


//**********************************************************************************************
//* POP3 class can be used in two ways:
//*
//* 1) 	Retrieving the whole mailbox, using POPMail::ReadInBox() method.
//* 	ReadInbox() method has two argument: 
//*
//*		mailbox:  	a reference to Vector<EMail> 
//		length:	 	specifies the length of the mail body to be downloaded.
//					length < 0: mails will be downloaded at full length.
//					length = 0; only mail headers will be downloaded.	
//					length < 0; [lenght] lines of the mail will be downloaded.				
//
//		returncode: TRUE if successfull, FALSE if failed.
 	
void InboxReaderThread(MessageInfo mi) 
{
	Vector<EMail> mailbox;
	POP3Mail pop3;
	
	 pop3.SSL(mi.ssl)
		.APOP(mi.apop)	
		.Host(mi.host)
		.Port(mi.port)
		.User(mi.user)
		.Pass(mi.pass)
		.Transcript(mi.transcript);


	if(!mi.gui->terminated) {
		if(!pop3.ReadInbox(mailbox, mi.full ? -1 : 0)) {
			String error;
			error 	<<  t_("POP3Mail Error: ")
				 	<<  pop3.GetError() << "\r\n";	
			PostCallback(callback2(mi.gui, &MailReader::Log, error, 1));
		}
		else {
			PostCallback(callback2(mi.gui, &MailReader::ShowInbox, mailbox, pop3.GetTranscript()));

		}
	}
	AtomicDec(mi.gui->threads);
}

//**********************************************************************************************
//* 2) Programmer decides the steps to be taken. (allows full control on POP3Mail class).
//*
//*	There are two requirements if this path is chosen:
//*
//* 	a) overall operation must be done inside a try-catch nest.
//*		b) overall operation must start with POP3Mail::Connect() 
//*		and (if successfull) end with POP3Mail::Disconnect()
//*
//*
		
void MailReaderThread(MessageInfo mi)
{
	POP3Mail pop3;
	EMail mail;

	pop3.SSL(mi.ssl)
		.APOP(mi.apop)
		.Transcript(mi.transcript);

	if(!mi.gui->terminated) {		
		if(!pop3.IsConnected()) {
			try {
				// Connect and Login to POP3 server.
				pop3.Connect(mi.host, mi.port, mi.user, mi.pass, mi.ssl, mi.apop);
				// See if we succesfully entered into TRANSACTION state.
				if(pop3.IsTransacting()) {
						// Read the specified mail.
						mail = pop3.GetMail(mi.index + 1);
						PostCallback(callback2(mi.gui, &MailReader::ShowMail, mail, pop3.GetTranscript()));	
				}
				// Disconnect from the server.
				pop3.Disconnect();
			}
			catch (Exc e) {
				String error;
				error	<<  t_("POP3Mail Error: ")
						<<  e << "\r\n";
				PostCallback(callback2(mi.gui, &MailReader::Log, error, 1));
			}	
		}
	}
	AtomicDec(mi.gui->threads);	
}

void MailReader::ReadInbox()
{
	MessageInfo mi;
	mi.host = sdlg.host;
	mi.port = sdlg.port;
	mi.user = sdlg.user;
	mi.pass = sdlg.pass;
	mi.full = sdlg.full;
	mi.transcript = sdlg.transcript;
	mi.apop = sdlg.apop;
	mi.ssl	= sdlg.ssl ? MailSocket::SSL : (sdlg.tls ? MailSocket::TLS : MailSocket::NOSSL);
	mi.gui	 = this;
	AtomicInc(threads);
	Thread().Run(callback1(InboxReaderThread, mi));
}

void MailReader::ReadMail()
{
	int n = list.GetCursor();
	if(n < 0)
		return;
	MessageInfo mi;
	mi.host = sdlg.host;
	mi.port = sdlg.port;
	mi.user = sdlg.user;
	mi.pass = sdlg.pass;
	mi.full = sdlg.full;
	mi.transcript = sdlg.transcript;
	mi.apop = sdlg.apop;
	mi.ssl	= sdlg.ssl ? MailSocket::SSL : (sdlg.tls ? MailSocket::TLS : MailSocket::NOSSL);
	mi.index = n;
	mi.gui	 = this;
	AtomicInc(threads);
	Thread().Run(callback1(MailReaderThread, mi));
}

void MailReader::ShowInbox(Vector<EMail>& inbox, String& log)
{
	list.Clear();
	for(int i = 0; i < inbox.GetCount(); i++) {
		list.Add(i + 1, 
			inbox[i].Subject(), 
			inbox[i].From());
	}
	Log(log);
}

void MailReader::ShowMail(EMail& mail, String& log)
{
	String header, body;
	mdlg.body.Clear();
	header.Clear();
	body.Clear();
	CtrlLayout(mdlg, mail.Subject());
	
	mdlg.subject 	= mail.Subject();
	mdlg.from		= mail.From();
	mdlg.to			= mail.To();
	mdlg.date		= mail.Date();
	mdlg.body.Set(mail.Body());

	if(!mdlg.IsOpen()) {
		mdlg.Sizeable()
			.Zoomable()
			.Execute();
	}
	Log(log);
}

void MailReader::SetToolBar(Bar& bar)
{
	bar.Add(MailImages::Envelope(), THISBACK(ReadInbox)).Text(t_("Read Inbox"));
	bar.Separator();
	bar.Add(MailImages::Console(), THISBACK(Console)).Text(t_("Open Debog/Log Console"));
	bar.Add(MailImages::Settings(), THISBACK(Settings)).Text(t_("Open Settings Window"));
	bar.Separator();
	bar.Add(MailImages::About(), THISBACK(About));
}

void MailReader::MainMenu(Bar& bar)
{
	bar.Add(t_("File"), THISBACK(FileMenu));
	bar.Add(t_("Help"), THISBACK(HelpMenu));	
}

void MailReader::FileMenu(Bar& bar)
{
	bar.Add(t_("Read Inbox..."), MailImages::Envelope(), THISBACK(ReadInbox));
	bar.Separator();
	bar.Add(t_("Settings"), MailImages::Settings(), THISBACK(Settings));
	bar.Separator();
	bar.Add(t_("Debug Console"),	MailImages::Console(), THISBACK(Console));
	bar.Separator();
	bar.Add(t_("Exit"), THISBACK(About));
}

void MailReader::HelpMenu(Bar& bar)
{
	bar.Add(t_("About"), MailImages::About(), THISBACK(About));
}

void MailReader::Settings()
{
	if(!sdlg.IsOpen()) {
		CtrlLayout(sdlg, t_("Account and Server Settings"));
		sdlg.Icon(MailImages::Settings());
		sdlg.Open(this);
	}
}

void MailReader::Console()
{
	if(!cdlg.IsOpen()) {
		CtrlLayout(cdlg, t_("Debug/Log Console"));
		cdlg.Icon(MailImages::Console());
		cdlg.Open(this);
		cdlg.Sizeable();
	}
}

void MailReader::About() 
{
	PromptOK("[A3 Simple POP3 Mail Reader]&Using [*^www://upp.sf.net^ Ultimate`+`+] technology.");
}

void MailReader::Log(String& s, bool error)
{
	 cdlg.console.Set(s); 
	if(error)
		Console();
}

void MailReader::Serialize(Stream& s)
{
	int version = 0;
	s / version;
	s % sdlg.host % sdlg.user % sdlg.pass;
	s % sdlg.apop % sdlg.transcript % sdlg.full;
	s % sdlg.port;
}

MailReader::MailReader()
{
	CtrlLayout(*this, "Simple Mail Reader Example.");
	Sizeable().Zoomable();
	Icon(MailImages::Envelope(), MailImages::Envelope());
	AddFrame(menu);
	AddFrame(TopSeparatorFrame());
	AddFrame(tool);
	menu.Set(THISBACK(MainMenu));
	tool.Set(THISBACK(SetToolBar));
	
	list.OddRowColor().EvenRowColor();
	list.AddColumn(t_(""));
	list.AddColumn(t_("Subject"));
	list.AddColumn(t_("From"));
	list.ColumnWidths("20 250 100");
	list.WhenLeftDouble = THISBACK(ReadMail);
	
	
	terminated 	= 0;
	threads		= 0;		
}

MailReader::~MailReader()
{
	terminated = 1;
	while(threads)
		Sleep(10);

}

GUI_APP_MAIN
{
	MailReader reader;
	LoadFromFile(reader);
	reader.Run();
	StoreToFile(reader);
}

